﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/glue/GlueRequest.h>
#include <aws/glue/Glue_EXPORTS.h>
#include <aws/glue/model/DatabaseInput.h>

#include <utility>

namespace Aws {
namespace Glue {
namespace Model {

/**
 */
class UpdateDatabaseRequest : public GlueRequest {
 public:
  AWS_GLUE_API UpdateDatabaseRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateDatabase"; }

  AWS_GLUE_API Aws::String SerializePayload() const override;

  AWS_GLUE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The ID of the Data Catalog in which the metadata database resides. If none is
   * provided, the Amazon Web Services account ID is used by default.</p>
   */
  inline const Aws::String& GetCatalogId() const { return m_catalogId; }
  inline bool CatalogIdHasBeenSet() const { return m_catalogIdHasBeenSet; }
  template <typename CatalogIdT = Aws::String>
  void SetCatalogId(CatalogIdT&& value) {
    m_catalogIdHasBeenSet = true;
    m_catalogId = std::forward<CatalogIdT>(value);
  }
  template <typename CatalogIdT = Aws::String>
  UpdateDatabaseRequest& WithCatalogId(CatalogIdT&& value) {
    SetCatalogId(std::forward<CatalogIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the database to update in the catalog. For Hive compatibility,
   * this is folded to lowercase.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  UpdateDatabaseRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A <code>DatabaseInput</code> object specifying the new definition of the
   * metadata database in the catalog.</p>
   */
  inline const DatabaseInput& GetDatabaseInput() const { return m_databaseInput; }
  inline bool DatabaseInputHasBeenSet() const { return m_databaseInputHasBeenSet; }
  template <typename DatabaseInputT = DatabaseInput>
  void SetDatabaseInput(DatabaseInputT&& value) {
    m_databaseInputHasBeenSet = true;
    m_databaseInput = std::forward<DatabaseInputT>(value);
  }
  template <typename DatabaseInputT = DatabaseInput>
  UpdateDatabaseRequest& WithDatabaseInput(DatabaseInputT&& value) {
    SetDatabaseInput(std::forward<DatabaseInputT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_catalogId;
  bool m_catalogIdHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  DatabaseInput m_databaseInput;
  bool m_databaseInputHasBeenSet = false;
};

}  // namespace Model
}  // namespace Glue
}  // namespace Aws
